import tkinter as tk
from tkinter import filedialog

from bebra_interpreter import interpret_bebra_generator
from bebra_highlighter import apply_highlighting, setup_tags

class BebraEditor:
    def __init__(self, root):
        self.root = root
        self.root.title("Bebra Editor")
        self.file_path = None

        self.text = tk.Text(root, wrap="word", font=("Courier New", 12), height=20)
        self.text.pack(fill="both", expand=True)

        self.text.bind("<KeyRelease>", lambda event: apply_highlighting(self.text))
        setup_tags(self.text)

        self.create_menu()
        self.create_buttons()

        self.console_output = tk.Text(root, height=10, bg="#111", fg="#0f0", insertbackground="#0f0", font=("Courier New", 10))
        self.console_output.pack(fill="x")
        self.console_output.insert("end", ">> Консоль готова\n")
        self.console_output.config(state="disabled")

    def create_menu(self):
        menu_bar = tk.Menu(self.root)
        file_menu = tk.Menu(menu_bar, tearoff=0)
        file_menu.add_command(label="New File", command=self.new_file)
        file_menu.add_command(label="Open File", command=self.open_file)
        file_menu.add_command(label="Save File", command=self.save_file)
        file_menu.add_command(label="Save As...", command=self.save_as)
        file_menu.add_separator()
        file_menu.add_command(label="Exit", command=self.root.quit)
        menu_bar.add_cascade(label="File", menu=file_menu)
        self.root.config(menu=menu_bar)

    def create_buttons(self):
        button_frame = tk.Frame(self.root)
        button_frame.pack(fill="x")
        run_button = tk.Button(button_frame, text="▶ Run", command=self.run_code, bg="#4CAF50", fg="white")
        run_button.pack(side="left", padx=5, pady=5)

    def new_file(self):
        self.file_path = None
        self.text.delete("1.0", tk.END)
        self.root.title("Bebra Editor - Untitled")

    def open_file(self):
        path = filedialog.askopenfilename(filetypes=[("Bebra files", "*.bebra"), ("All files", "*.*")])
        if path:
            with open(path, "r", encoding="utf-8") as f:
                content = f.read()
                self.text.delete("1.0", tk.END)
                self.text.insert(tk.END, content)
            self.file_path = path
            self.root.title(f"Bebra Editor - {path}")
            apply_highlighting(self.text)

    def save_file(self):
        if self.file_path:
            with open(self.file_path, "w", encoding="utf-8") as f:
                f.write(self.text.get("1.0", tk.END))
        else:
            self.save_as()

    def save_as(self):
        path = filedialog.asksaveasfilename(defaultextension=".bebra",
                                            filetypes=[("Bebra files", "*.bebra"), ("All files", "*.*")])
        if path:
            with open(path, "w", encoding="utf-8") as f:
                f.write(self.text.get("1.0", tk.END))
            self.file_path = path
            self.root.title(f"Bebra Editor - {path}")

    def log_to_console(self, message):
        self.console_output.config(state="normal")
        self.console_output.insert(tk.END, message + "\n")
        self.console_output.see(tk.END)
        self.console_output.config(state="disabled")

    def show_popup(self, message):
        popup = tk.Toplevel(self.root)
        popup.title("Bebra Output")
        popup.geometry("300x100")
        label = tk.Label(popup, text=message, font=("Arial", 12))
        label.pack(expand=True, padx=10, pady=10)

    def run_code(self):
        code = self.text.get("1.0", tk.END)
        self.console_output.config(state="normal")
        self.console_output.delete("1.0", tk.END)
        self.console_output.config(state="disabled")

        self.interpreter_gen = interpret_bebra_generator(code, log=self.log_to_console, show_popup=self.show_popup)
        self.process_interpreter_step()

    def process_interpreter_step(self):
        try:
            cmd, arg = next(self.interpreter_gen)
            if cmd == "continue":
                self.root.after(10, self.process_interpreter_step)
            elif cmd == "wait":
                ms = int(arg * 1000)
                self.root.after(ms, self.process_interpreter_step)
            elif cmd == "error":
                self.log_to_console(f"❌ {arg}")
            # убираем сообщение о выполнении:
            # elif cmd == "done":
            #     self.log_to_console("✅ Выполнение завершено")
        except StopIteration:
            pass  # Просто молча завершаем

if __name__ == "__main__":
    root = tk.Tk()
    app = BebraEditor(root)
    root.mainloop()
